
// Copyright 1992, David Perelman-Hall & Jamshid Afshar

#include <assert.h>
#include "misc.h"
#include "edge.h"


Edge combine( const Edge& edge1, const Edge& edge2 )
{
   assert( edge1.canCombineWith( edge2 ));
   const Edge *ap, *ip;

   if( edge1.isActive() ){
      ap = &edge1;
      ip = &edge2;
   }
   else{
      ip = &edge1;
      ap = &edge2;
   }

   Tree tmp( ap->_tree );
   tmp.add_child( ip->_tree );


   return Edge( ap->_start, ip->_finish, tmp, ap->_toFind.rest() );
}


ostream& operator << ( ostream& os, const Edge& edge )
{
   os << "<" << edge.start() << "," << edge.finish() << ","
      << edge.label() << ",";
   os << edge.found() << "." << edge.toFind() << ">";

   return os;
}


// constructor
Edge::Edge( int start, int finish, const Category& label )
   : _start(start), _finish(finish), _tree(label) {}


// constructor
Edge::Edge( int start, int finish, const Category& label,
               const Category_Sequence& toFind )
   : _start(start), _finish(finish), _tree(label), 
     _toFind(toFind) {}

// constructor
Edge::Edge( int start, int finish, const Tree& tree,
            const Category_Sequence& toFind )
   : _start(start), _finish(finish), _tree(tree), 
     _toFind(toFind) {}

// copy constructor
Edge::Edge( const Edge& edge )
   : _start(edge._start), _finish(edge._finish), _tree(edge._tree), 
     _toFind(edge._toFind) {}


// assignment operator
void Edge::operator = ( const Edge& edge )
{
   _start = edge._start;
   _finish = edge._finish;
   _tree = edge._tree;
   _toFind = edge._toFind; 
}

// boolean equal
bool Edge::operator == ( const Edge& edge ) const
{
   return(  _start == edge._start  &&
            _finish == edge._finish  &&
            _tree == edge._tree  &&
            _toFind == edge._toFind );
}   

bool Edge::canCombineWith( const Edge& edge ) const
{
   const Edge *ap, *ip;

   if( isActive() ){
      ip = &edge;
      ap = this;
   }
   else{
      ip = this;
      ap = &edge;
   }

   if (!ap->isActive() || ip->isActive()) {
      return FALSE;
   }      
   else {
      Category cat = ap->_toFind.first();
      return (ip->_start == ap->_finish ) && ( ip->_tree.cat() == cat );
   }
}
      
// constructor
Edge_List::Edge_List()
   : _firstNode(NULL){}

   
// copy constructor
Edge_List::Edge_List( const Edge_List& edge_List )
   : _firstNode(NULL)
{
   *this = edge_List;
}


Edge_List::~Edge_List()
{
   clear();
}

void Edge_List::push( const Edge& edge )
{
   edgeNode *newnode = new edgeNode( edge, _firstNode );
   _firstNode = newnode;
}


Edge Edge_List::pop()
{
   assert( _firstNode != NULL );

   Edge e = _firstNode->_edge;  // call to copy constructor
   edgeNode *eptr = _firstNode;
   _firstNode = _firstNode->_next;
   delete eptr;
   return e;
}

// assignment operator 
void Edge_List::operator = ( const Edge_List& edge_List )
{
   // delete allocated nodes
   clear();

   const edgeNode *enp = edge_List._firstNode;
   edgeNode **thisenpp = &_firstNode;
   while ( enp != NULL ){
      *thisenpp = new edgeNode(enp->_edge, NULL);
      thisenpp = &((*thisenpp)->_next);
      enp = enp->_next;
   }
}

void Edge_List::clear()
{
   while ( !isEmpty() )
      pop();
}


bool Edge_List::isMember( const Edge& edge ) const
{
   const edgeNode *enp = _firstNode;
   while ( enp != NULL ){
      if( enp->_edge == edge )
         return TRUE;
      enp = enp->_next;
   }
   return FALSE;
}

bool Edge_List::isEmpty() const
{
   return _firstNode == NULL;
}

ostream& operator << ( ostream& os, const Edge_List& list )
{
   const edgeNode *enp = list._firstNode;
   while ( enp != NULL ){
      cout << enp->_edge << "\n";
      enp = enp->_next;
   }
   return os;
}

